<?php

namespace App\Http\Controllers\WebAdmin;

use App\Events\MailSendEvent;
use App\Http\Controllers\Controller;
use App\Http\Requests\InstructorStoreRequest;
use App\Models\Instructor;
use App\Models\User;
use App\Repositories\AccountActivationRepository;
use App\Repositories\UserRepository;
use App\Rules\PhoneNumber;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoginController extends Controller
{
    public function index()
    {
        if (Auth::check()) {
            if (Auth::user()->is_admin && Auth::user()->hasRole('admin')) {
                return redirect()->route('admin.dashboard');
            }
            if (Auth::user()->hasRole('instructor')) {
                return redirect()->route('instructor.dashboard');
            }
        }
        return view('auth.login');
    }

    public function authenticate(Request $request): RedirectResponse
    {
        $credentials = $request->validate([
            'email' => ['required'],
            'password' => ['required'],
        ]);

        if (Auth::attempt($credentials)) {
            $user = UserRepository::query()->where('email', $request->email)->first();

            Auth::login($user, true);
            $request->session()->regenerate();

            if ($user->is_admin && $user->hasRole('admin')) {
                return redirect()->route('admin.dashboard');
            }
            return redirect()->route('instructor.dashboard');
        }

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function instructorRegister()
    {
        return view('auth.instructor.register');
    }

    public function instructorAuthenticate(Request $request)
    {

        $credentials = $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'phone' => ['required', 'unique:users,phone',  new PhoneNumber()],
            'password' => 'required|min:8|regex:/[A-Z]/|regex:/[a-z]/|regex:/[0-9]/|regex:/[@$!%*?&]/|confirmed',
        ]);


        $credentials['password'] = bcrypt($credentials['password']);
        $user = User::create($credentials);
        $user->assignRole('instructor');

        Instructor::create([
            'user_id' => $user->id,
            'title' => 'Simple Instructor',
            'created_at' => now(),
        ]);

        $user->sendEmailVerificationNotification();

        return redirect()->route('admin.login')->with('account-created', 'Account created successfully, please check your email');
    }

    public function logout(Request $request): RedirectResponse
    {
        Auth::logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }
}
